module String = 

  (* bee_* functions are runtime primitives *)
  let length = bee_string_length
  let build = bee_string_build (* this may fail with an out_of_memory, but in our current system such errors are fatal *)
  let blit = bee_string_blit (* source, destination *)
  let get = bee_string_get
  let set = bee_string_set
  
  let copy s =
    let len = length s in
    let r = build len in
    blit s 0 r 0 len;
    r
  
  let sub s offset len = 
    if offset + len > length s then (* TODO: can be removed if rich types *)
      print_string "invalid arguments in String.sub\n";
      print_string "string length: "; print_int (length s); print_string "\n";
      print_string "offset: "; print_int offset; print_string "\n";
      print_string "len: "; print_int len; print_string "\n";
      error 1
    else
      (let r = build len in
       blit s offset r 0 len;
       r)
    
  let prefix s n = sub s 0 n

  let suffix s i = sub s i (length s - i)
       
  let concat s1 s2 = 
    let s = build (length s1 + length s2) in 
    blit s1 0 s 0 (length s1);
    blit s2 0 s (length s1) (length s2);
    s
    
  let (^) = concat
       
  let join l s = (* TODO: not the most efficient *)
    case l 
     | [] -> ""
     | t :: q -> List.fold (fun b a -> b ^ s ^ a) t q
    end

  let trim s = 
    let characterstotrim = [' '; ':'; '-'; '_'; ';'; ','; '.'; '+'; ] in 
    let rec trimleft s1 = 
      if s1 <> "" && List.mem (get s1 0) characterstotrim then trimleft (sub s1 1 (length s1 - 1)) else s1 in 
    let rec trimright s1 = 
      if s1 <> "" && List.mem (get s1 (length s1 - 1)) characterstotrim then trimright (sub s1 0 (length s1 - 1)) else s1 in 
    trimleft (trimright s)
  
  
  let rec findchar s c p = 
    if p >= length s 
    then None
    else
      if s.[p] = c 
      then Some p 
      else findchar s c (p + 1)
       
  (* a naive string search algorithm TODO: tail recursion not detected by compiler *)
  (* TODO: add a non naive, more space intensive version for string search *)
  let rec indexofuntilfrom firstchar s1 s2 n1 n2 frompos =
    let start = ref frompos in
  
    let rec loop1 () = (* this loop goes to the first matching character *)
      if !start < n2 && get s2 !start <> firstchar then 
        start := !start + 1;
        loop1 () 
      else () in 
    loop1 ();
  
    if !start >= n2 then 
      None
    else 
      let ndx = ref 1 in
      let rec loop2 () = 
        if !ndx < n1 && !ndx + !start < n2 && get s1 !ndx = get s2 (!ndx + !start) then
          ndx := !ndx + 1;
          loop2 ()
        else () in 
      loop2 ();
      if !ndx = n1 
      then Some !start
      else indexofuntilfrom firstchar s1 s2 n1 n2 (!start + 1)
  
  let indexoffrom s1 s2 p = indexofuntilfrom (get s1 0) s1 s2 (length s1) (length s2) p

  let indexof s1 s2 = indexofuntilfrom (get s1 0) s1 s2 (length s1) (length s2) 0

  let split s c = 
    let rec split_aux s c pos l = 
      case findchar s c pos
       | None -> (sub s pos (length s - pos)) :: l
       | Some p -> split_aux s c (p + 1) ((sub s pos (p - pos)) :: l)
      end
     in 
    List.rev (split_aux s c 0 []) (* rev cost in space should be dominated by strings cost *)
        
  let extract_between s start_mark end_mark = 
    match indexoffrom start_mark s 0 with  
        Some pos -> 
          match indexoffrom end_mark s (pos + length start_mark) with
              Some pos1 -> sub  s  (pos + length start_mark)  (pos1 - (pos + length start_mark))
            | None -> ""
          end
      | None -> ""
    end 
    
  let extract_between_from s start_mark end_mark frompos = 
    match indexoffrom start_mark s frompos with  
        Some pos -> 
          match indexoffrom end_mark s (pos + length start_mark) with
              Some pos1 -> (sub s (pos + length start_mark) (pos1 - (pos + length start_mark)), pos1 + length end_mark)
            | None -> ("", -1)
          end
      | None -> ("", -1)
    end 
    
  (* convert a string in decimal format to an integer
     WARNING: returns 0 if the string cannot be converted to an integer *)
  let to_int s = 
    let res = ref 0 in 
    for  0  (length s - 1)
      (fun i -> 
          let digit_code = int_of_char (get s i) in 
          let digit = digit_code - 48 in 
          if digit >= 0 && digit < 10 then
            res := !res * 10 + digit
          else 
            ());
    !res
    
endmodule

(* declarations that are allowed to escape the module scope *)
let (^) = String.concat


